/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Collections;
using System.Collections.Specialized;
using System.ComponentModel;


using Borland.Eco.Internal.DefaultImpl;
using Borland.Eco.DataRepresentation;
using Borland.Eco.Persistence.Configuration;
using Borland.Eco.Persistence.Connection;
using Borland.Eco.Persistence.ORMapping;
using Borland.Eco.Services;
using Borland.Eco.Globalization;

namespace Borland.Eco.Persistence
{

	public abstract class PersistenceMapperDb: AbstractPersistenceMapperDb, IPersistenceMapperDb
	{
		private SqlDatabaseConfig m_SqlDatabaseConfig = new SqlDatabaseConfig();

		private IORMappingProvider m_DefaultOldMappingProvider = new DbMappingProvider();
		private DefaultORMappingBuilder m_DefaultNewMappingProvider = new DefaultORMappingBuilder();

		protected override void SetCompatibilityMode(EcoCompatibilityMode mode)
		{
			base.SetCompatibilityMode(mode);
			m_DefaultNewMappingProvider.SetEcoCompatibilityMode(mode);
		}

		private IORMappingProvider m_OldMappingProvider;
		private IORMappingProvider m_NewMappingProvider;

		private readonly ConnectionPool m_ConnectionPool;

		protected PersistenceMapperDb()
		{
			m_ConnectionPool = new ConnectionPool(this);
		}

		[Browsable(false)]
		public abstract bool HasConnection { get; }

		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		[Browsable(false)]
		[Obsolete("Active property is obsolete. Setting or getting this property has no effect. Please remove code line.")]
		public bool Active { set {} }

		[Browsable(false)]
		public ConnectionPool ConnectionPool
		{
			get { return m_ConnectionPool; }
		}

		[LocalizableCategory(typeof(PersistenceStringRes), "sCategoryMapping")]
		[LocalizableDescription(typeof(PersistenceStringRes), "sPropertyOldMappingProvider")]
		[DefaultValue(null)]
		public IORMappingProvider OldMappingProvider
		{
			get { return m_OldMappingProvider; }
			set
			{
				EnsureNotActive();
				m_OldMappingProvider = value;
			}
		}

		///<summary>
		/// This property returns the OldMappingProvider that will actually be used in runtime. If the OldMappingProvider
		/// property is not set, this property will return a DbMappingProvider
		///</summary>
		[Browsable(false)]
		public IORMappingProvider EffectiveOldMappingProvider
		{
			get
			{
				return (m_OldMappingProvider != null) ? m_OldMappingProvider : m_DefaultOldMappingProvider;
			}
		}

		[LocalizableCategory(typeof(PersistenceStringRes), "sCategoryMapping")]
		[LocalizableDescription(typeof(PersistenceStringRes), "sPropertyNewMappingProvider")]
		[DefaultValue(null)]
		public IORMappingProvider NewMappingProvider
		{
			get { return m_NewMappingProvider; }
			set
			{
				EnsureNotActive();
				m_NewMappingProvider = value;
			}
		}

		///<summary>
		/// This property returns the NewMappingProvider that will actually be used in runtime. If the NewMappingProvider
		/// property is not set, this property will return a DefaultOrMappingProvider
		///</summary>
		[Browsable(false)]
		public IORMappingProvider EffectiveNewMappingProvider
		{
			get
			{
				return (m_NewMappingProvider != null) ? m_NewMappingProvider : m_DefaultNewMappingProvider;
			}
		}

#region Schema manipulation/inspection methods
		///<summary>
		/// This method can be called in runtime to create a database schema in the database refered to by the persistence mapper.
		/// DefaultCleanPsConfig(false) will be used as the IConfigureCleanPs instance.
		///</summary>
		///<exception cref="ArgumentNullException">Thrown if typeSystemService is null</exception>
		public void CreateDataBaseSchema(ITypeSystemService typeSystemService)
		{
			CreateDataBaseSchema(typeSystemService, new DefaultCleanPsConfig(false));
		}
		///<summary>
		///Returns the script required to create the schema for the current model.
		///</summary>
		///<exception cref="ArgumentNullException">Thrown if typeSystemService is null</exception>
		public StringCollection CreateSchemaScript(ITypeSystemService typeSystemService)
		{
			if (typeSystemService == null) throw new ArgumentNullException("typeSystemService"); // do not localize
			EffectiveNewMappingProvider.Initialize(typeSystemService.TypeSystem, null, SqlDatabaseConfig, true);

			IDatabase db = m_ConnectionPool.RetrieveClosedDatabaseConnection(true);

			try
			{
				EffectiveNewMappingProvider.Initialize(typeSystemService.TypeSystem, null, db.Config, true);
				EffectiveNewMappingProvider.Mapping.Validate(db.Config, typeSystemService.TypeSystem);

				string dbName = GetNewDbName();
				return PersistenceHandleDb.CreateSchemaScript(typeSystemService, GetDatabaseCollection(db, dbName), dbName, EffectiveNewMappingProvider, EffectiveOldMappingProvider);
			}
			finally
			{
				if (db != null)
				{
					m_ConnectionPool.ReturnDatabaseConnection(db);
				}
			}

		}

		private string GetRuntimeDbName()
		{
			EnsureSingleDb(EffectiveRunTimeMappingProvider);
			return EffectiveRunTimeMappingProvider.Mapping.Databases.Any.Name;
		}

		private string GetNewDbName()
		{
			EnsureSingleDb(EffectiveNewMappingProvider);
			return EffectiveNewMappingProvider.Mapping.Databases.Any.Name;
		}



		///<summary>
		///Validates that the schema matches the model. remedies will contain the SQL statements to
		///rectify the problems in the schema.
		///</summary>
		public bool ValidateSchemaStructure(StringCollection remedies, ITypeSystemService typeSystemService)
		{
			IDatabase database = null;
			bool result = false;
			try
			{
				database = m_ConnectionPool.RetrieveDatabaseConnection(true);
				string dbName = GetRuntimeDbName();

				result = PersistenceHandleDb.ValidateSchemaStructure(remedies, typeSystemService, GetDatabaseCollection(database, dbName), dbName, EffectiveNewMappingProvider);
			}
			finally
			{
				if (database != null)
				{
					database.Close();
					m_ConnectionPool.ReturnDatabaseConnection(database);
				}
			}
			return result;
		}
		///<summary>
		///Validates that the schema matches the model. remedies will contain the SQL statements to
		///rectify the problems in the data structure.
		///</summary>
		public bool ValidateDataStructure(StringCollection remedies, ITypeSystemService typeSystemService)
		{
			IDatabase database = null;
			bool result = false;
			try
			{
				database = m_ConnectionPool.RetrieveDatabaseConnection(true);
				string dbName = GetRuntimeDbName();
				result = PersistenceHandleDb.ValidateDataStructure(remedies, typeSystemService, GetDatabaseCollection(database, dbName), dbName, EffectiveNewMappingProvider);
			}
			finally
			{
				if (database != null)
				{
					database.Close();
					m_ConnectionPool.ReturnDatabaseConnection(database);
				}
			}
			return result;
		}

		///<summary>
		/// This method can be called in runtime to create a database schema in the database refered to by the persistence mapper
		///<exception cref="ArgumentNullException">Thrown if typeSystemService is null</exception>
		///<exception cref="ArgumentNullException">Thrown if configureCleanPS is null</exception>
		///</summary>
		public void CreateDataBaseSchema(ITypeSystemService typeSystemService, IConfigureCleanPS configureCleanPS)
		{
			if (configureCleanPS == null) throw new ArgumentNullException("configureCleanPS"); // do not localize
			if (typeSystemService == null) throw new ArgumentNullException("typeSystemService"); // do not localize

			IDatabase database = null;
			try
			{
				database = m_ConnectionPool.RetrieveDatabaseConnection(true);
				EffectiveNewMappingProvider.Initialize(typeSystemService.TypeSystem, null, database.Config, true);
				EffectiveNewMappingProvider.Mapping.Validate(database.Config, typeSystemService.TypeSystem);
				EffectiveOldMappingProvider.Initialize(null, null, database.Config, false);
				string dbName = GetNewDbName();

				PersistenceHandleDb.CreateDataBaseSchema(typeSystemService, GetDatabaseCollection(database, dbName), dbName, EffectiveNewMappingProvider, EffectiveOldMappingProvider, configureCleanPS);
			}
			finally
			{
				if (database != null)
				{
					database.Close();
					m_ConnectionPool.ReturnDatabaseConnection(database);
				}
			}
		}
		///<summary>
		///<p>Applies the statements in sqlSqript. Script executed as ExecuteSql, and must not return an answer set.
		///If SqlDatabaseConfig.AllowMetadataChangesInTransaction is true, execution will be wrapped in
		///a transaction.</p>
		///Sql-exceptions will be rethrown.
		///<exception cref="ArgumentNullException">Thrown if sqlScript is null.</exception>
		///<exception cref="InvalidOperationException">Thrown if SqlDatabaseConfig is not configured.</exception>
		///</summary>
		public void ApplyScript(StringCollection sqlScript)
		{
			if (sqlScript == null) throw new ArgumentNullException("sqlScript"); // no not localize
			if (!SqlDatabaseConfig.ConfigValid)
				throw new InvalidOperationException(PersistenceStringRes.sSqlDatabaseConfigNotValid);
			IDatabase database = null;
			try
			{
				database = m_ConnectionPool.RetrieveDatabaseConnection(true);
				if (SqlDatabaseConfig.AllowMetadataChangesInTransaction)
					database.StartTransaction();
				try
				{
					IExecQuery query = database.GetExecQuery();
					try
					{
						foreach (string sqlStatement in sqlScript)
						{
							query.AssignSqlText(sqlStatement);
							query.ExecSql();
						}
					}
					finally
					{
						database.ReleaseExecQuery(query);
					}
					if (database.InTransaction)
						database.Commit();
				}
				catch
				{
					if (database.InTransaction)
						database.RollBack();
					throw;
				}
			}
			finally
			{
				if (database != null)
				{
					database.Close();
					m_ConnectionPool.ReturnDatabaseConnection(database);
				}
			}
		}

#endregion
		/// <summary>
		/// This method will store the mapping information that is needed by DbEvolution in a table in the database.
		/// Normally this information is stored when CreateDatabaseSchema is called, but for databases created with Eco1,
		/// this method must be called prior to changing the model.
		///<exception cref="ArgumentNullException">Thrown if typeSystemService is null</exception>
		/// </summary>
		public void SaveMappingInfo(ITypeSystemService typeSystemService)
		{
			if (typeSystemService == null) throw new ArgumentNullException("typeSystemService"); // do not localize
			IDatabase database = null;
			try
			{
				database = m_ConnectionPool.RetrieveDatabaseConnection(true);
				EffectiveRunTimeMappingProvider.Initialize(typeSystemService.TypeSystem, database, SqlDatabaseConfig, false);
				EffectiveOldMappingProvider.SaveMappingInfo(database, SqlDatabaseConfig, EffectiveRunTimeMappingProvider.Mapping);
			}
			finally
			{
				if (database != null)
				{
					database.Close();
					m_ConnectionPool.ReturnDatabaseConnection(database);
				}
			}
		}
		/// <summary>
		/// Utility function that will examine if the database contains a table with the specified name.
		/// </summary>
		public bool DatabaseHasTable(string tableName)
		{
			IDatabase database = null;
			bool res = false;
			try
			{
				database = m_ConnectionPool.RetrieveDatabaseConnection(true);
				res = database.AllTableNames(tableName, false).Count == 1;
			}
			finally
			{
				if (database != null)
				{
					database.Close();
					m_ConnectionPool.ReturnDatabaseConnection(database);
				}
			}
			return res;
		}
		/// <summary>
		/// Retrieves an evolutor.
		///<exception cref="ArgumentNullException">Thrown if typeSystemService is null</exception>
		/// </summary>
		public IDBEvolutor GetEvolutor(ITypeSystemService typeSystemService)
		{
			if (typeSystemService == null) throw new ArgumentNullException("typeSystemService"); // do not localize
			IDatabase database = null;
			try
			{
				database = m_ConnectionPool.RetrieveDatabaseConnection(true);

				EffectiveOldMappingProvider.Initialize(typeSystemService.TypeSystem, database, database.Config, false);
				EffectiveNewMappingProvider.Initialize(typeSystemService.TypeSystem, database, database.Config, true);
				EffectiveNewMappingProvider.Mapping.Validate(database.Config, typeSystemService.TypeSystem);

				string dbName = GetNewDbName();

				return PersistenceHandleDb.GetEvolutor(typeSystemService, GetDatabaseCollection(database, dbName), dbName, EffectiveNewMappingProvider, EffectiveOldMappingProvider);
			}
			catch
			{
				if (database != null)
				{
					database.Close();
					m_ConnectionPool.ReturnDatabaseConnection(database);
				}
				throw;
			}
		}

		/// <summary>
		/// Returns an evolutor after use.
		///<exception cref="ArgumentNullException">Thrown if evolutor is null</exception>
		/// </summary>
		public void ReturnEvolutor(IDBEvolutor evolutor)
		{
			if (evolutor == null) throw new ArgumentNullException("evolutor"); // do not localize
			IDatabase database = evolutor.Database;
			if (database != null)
			{
				database.Close();
				m_ConnectionPool.ReturnDatabaseConnection(database);
			}
		}

		[DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
		[LocalizableCategory(typeof(PersistenceStringRes), "sCategoryPersistence")]
		[LocalizableDescription(typeof(PersistenceStringRes), "sPropertySqlDatabaseConfig")]
		public SqlDatabaseConfig SqlDatabaseConfig
		{
			get { return m_SqlDatabaseConfig; }
		}

		[Browsable(false)] // Backwards compatibility
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		[Obsolete()]
		public string ClockLogGranularity
		{
			set {}
		}

		[Browsable(false)] // Backwards compatibility
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		[Obsolete()]
		public bool EvolutionSupport // Allow old apps
		{
			set {}
		}

		/// <summary>
		/// <para>MaxOpenConnections is the maximum number of connections that will be open at once.</para>
		/// <para>Connections are only used when Eco is actually performing an operation in persistent storage
		/// So it can usually be signifcanty less that the number of EcoSpaces sharing the PersistenceMapper.</para>
		/// <para> if MaxOpenConnections is &lt; MaxPoolConnections then the pool wi never be fully used. </para>
		/// <para>If the MaxOpenConnections is is &gt; MaxPoolConnections, then the additiona connections wil be disposed after use, rather
		/// than returned to the pool </para>
		/// <para>If MaxOpenConnections is &gt; the connection will always be cloned, the
		/// actual connection instace connected to the PersistenceMapper will
		/// not be used as such.</para>
		/// </summary>
		[LocalizableCategory(typeof(PersistenceStringRes), "sCategoryPooling")]
		[LocalizableDescription(typeof(PersistenceStringRes), "sPropertyMaxOpenConnections")]
		[DefaultValue(1)]
		public int MaxOpenConnections
		{
			get { return m_ConnectionPool.MaxOpenConnections; }
			set
			{
				EnsureNotActive();
				m_ConnectionPool.MaxOpenConnections = value;
			}
		}

		/// <summary>
		/// <para>MaxPoolConnections is the maximum number of connections that will be pooled.</para>
		/// <para>C</para>
		/// </summary>
		[LocalizableCategory(typeof(PersistenceStringRes), "sCategoryPooling")]
		[LocalizableDescription(typeof(PersistenceStringRes), "sPropertyMaxOpenConnections")]
		[DefaultValue(1)]
		public int MaxPoolConnections
		{
			get { return m_ConnectionPool.MaxPoolConnections; }
			set
			{
				EnsureNotActive();
				m_ConnectionPool.MaxPoolConnections = value;
			}
		}

		public override void CloseConnections(bool force)
		{
			ConnectionPool.CloseConnections(force);
		}

		protected internal abstract IDatabase MakeDatabase();  // Reachable by connectionpool
		protected internal abstract void ReturnDatabase(IDatabase database); // Reachable by connectionpool

		private static IDatabaseCollection GetDatabaseCollection(IDatabase db, string name)
		{
			Borland.Eco.Persistence.Connection.DatabaseCollection databases = new Borland.Eco.Persistence.Connection.DatabaseCollection(new LocalTransactionManager());
			databases.Add(db, name);
			return databases;
		}

		private void EnsureSingleDb(IORMappingProvider mappingProvider)
		{
			if (mappingProvider.Mapping.Databases.Count == 0)
				throw new InvalidOperationException(PersistenceStringRes.sMappingDoesNotDefineAnyDatabase);
			if (mappingProvider.Mapping.Databases.Count > 1)
				throw new InvalidOperationException(PersistenceStringRes.sMappingDefinesMultipleDatabases);
		}

		///<exception cref="InvalidOperationException">Thrown if ActiveCount > 0.</exception>
		public override IPersistenceMapper GetPersistenceMapper(ITypeSystemService typeSystemService)
		{
			lock(this)
			{
				if ((PmapperWithDb == null) || (this.TypeSystemService != typeSystemService))
				{
					if (ActiveCount > 0)
						throw new InvalidOperationException(PersistenceStringRes.sPersistenceMapperInUseWithOtherTypeSystem);

					SetPersistenceMapperWithIDatabase(null);
					SetTypeSystemService(null);
					IDatabase database = null;
					try
					{
						database = m_ConnectionPool.RetrieveDatabaseConnection(true);
						SetTypeSystemService(typeSystemService);
						if (typeSystemService == null)
							throw new ArgumentNullException("typeSystemService");
						EffectiveRunTimeMappingProvider.Initialize(typeSystemService.TypeSystem, database, database.Config, false);
						EffectiveRunTimeMappingProvider.Mapping.Validate(database.Config, typeSystemService.TypeSystem);

						string dbName = GetRuntimeDbName();

						SetPersistenceMapperWithIDatabase(PersistenceHandleDb.GetPersistenceMapper(
							TypeSystemService,
							GetDatabaseCollection(database, dbName),
							EffectiveRunTimeMappingProvider.Mapping,
							VersionGranularity, OnGetCurrentTime));
					}
					finally
					{
						if (database != null)
							m_ConnectionPool.ReturnDatabaseConnection(database);
					}
				}
				ActiveCount++;
				return new PersistenceMapperAdapter(PmapperWithDb, this);
			}
		}

		public override void ReturnPersistenceMapper(IPersistenceMapper persistenceMapper)
		{
			lock(this)
			{
				ActiveCount--;
				if (ActiveCount == 0)
					CloseConnections(false);
			}
		}

		private sealed class PersistenceMapperAdapter: MarshalByRefObject, IPersistenceMapper
		{
			private IPersistenceMapperWithIDatabase m_Adaptee;
			private readonly PersistenceMapperDb m_PersistenceMapperDb;

			private IDatabase RetrieveDatabaseConnection()
			{
				return m_PersistenceMapperDb.ConnectionPool.RetrieveDatabaseConnection(false);
			}

			private IDatabaseCollection RetrieveDatabaseConnections()
			{
				string dbName = m_PersistenceMapperDb.GetRuntimeDbName();

				Borland.Eco.Persistence.Connection.DatabaseCollection collection = new Borland.Eco.Persistence.Connection.DatabaseCollection(new LocalTransactionManager());
				collection.Add(RetrieveDatabaseConnection(), dbName);
				return collection;
			}

			private void ReturnDatabaseConnections(IDatabaseCollection databases)
			{
				foreach (IDatabase db in databases)
					ReturnDatabaseConnection(db);
			}

			private void ReturnDatabaseConnection(IDatabase database)
			{
				m_PersistenceMapperDb.ConnectionPool.ReturnDatabaseConnection(database);
			}

			private IPersistenceMapperWithIDatabase Adaptee
			{
				get
				{
					if (m_Adaptee == null)
						throw new InvalidOperationException("Attempt to use deactivated PersistenceMapperAdapter"); // Do not localize
					return m_Adaptee;
				}
			}

			public PersistenceMapperAdapter(IPersistenceMapperWithIDatabase persistenceMapperWithIDatabase, PersistenceMapperDb persistenceMapperDb)
			{
				m_PersistenceMapperDb = persistenceMapperDb;
				m_Adaptee = persistenceMapperWithIDatabase;
			}

			public void DeActivate()
			{
				m_Adaptee = null;
			}

			#region IPersistenceMapper implementation

			void IPersistenceMapper.Fetch(ObjectIdList idList, out Datablock datablock, int[] memberIdList)
			{
				IDatabaseCollection databases = RetrieveDatabaseConnections();
				try
				{
					Adaptee.Fetch(databases, idList, out datablock, memberIdList);
					if (datablock == null)
						throw new InvalidOperationException();
					datablock.Strip();
				}
				finally
				{
					ReturnDatabaseConnections(databases);
				}
			}
			void IPersistenceMapper.FetchIDListWithCondition(AbstractCondition condition, out ObjectIdList result, int maxResults, int offset)
			{
				IDatabaseCollection databases = RetrieveDatabaseConnections();
				try
				{
					Adaptee.FetchIDListWithCondition(databases, condition, out result, maxResults, offset);
				}
				finally
				{
					ReturnDatabaseConnections(databases);
				}
			}
			void IPersistenceMapper.VersionForTime(DateTime clockTime, out int version)
			{
				IDatabase database = RetrieveDatabaseConnection();
				try
				{
					Adaptee.VersionForTime(database, clockTime, out version);
				}
				finally
				{
					ReturnDatabaseConnection(database);
				}
			}
			void IPersistenceMapper.TimeForVersion(int version, out DateTime clockTime)
			{
				IDatabase database = RetrieveDatabaseConnection();
				try
				{
					Adaptee.TimeForVersion(database, version, out clockTime);
				}
				finally
				{
					ReturnDatabaseConnection(database);
				}
			}

			bool IPersistenceMapper.SupportsSync
			{
				get { return m_PersistenceMapperDb.SyncHandler != null; }
			}

			void IPersistenceMapper.Update(Datablock datablock, UpdatePrecondition precondition, out IdTranslationList translationList, out int version, out SyncVersion syncVersion, out UpdateResult result)
			{
				IDatabaseCollection databases = RetrieveDatabaseConnections();
				try
				{
					Adaptee.Update(databases, datablock, precondition, out translationList, out version, out result, out syncVersion,
					 m_PersistenceMapperDb.SyncHandler);
				}
				finally
				{
					ReturnDatabaseConnections(databases);
				}
			}

			void IPersistenceMapper.GetChangesSince(SyncVersion syncVersion, SyncVersion[] excludeList, out DBChangeCollection changes, out SyncVersion lastSyncVersion)
			{
				if (m_PersistenceMapperDb.SyncHandler == null)
					throw new InvalidOperationException(PersistenceStringRes.sSyncNotActive);
    			changes = m_PersistenceMapperDb.SyncHandler.GetChangesSince(syncVersion, excludeList, out lastSyncVersion, m_PersistenceMapperDb.TypeSystemService.TypeSystem);
				}

			SyncVersion IPersistenceMapper.CurrentSyncVersion
			{
				get
				{
					if (m_PersistenceMapperDb.SyncHandler == null)
						throw new InvalidOperationException(PersistenceStringRes.sSyncNotActive);
					return m_PersistenceMapperDb.SyncHandler.CurrentSyncVersion();
				}
			}
			#endregion
		}
	}
}
